<?php

namespace App\Http\Controllers\Admin;

use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use App\Models\UserWallet;
use App\Models\User;
use Carbon\Carbon;
use Exception;
use App\Http\Helpers\Response;
use App\Models\UserNotification;
use App\Constants\NotificationConst;
use App\Events\User\NotificationEvent as UserNotificationEvent;
use App\Notifications\User\AddMoney\ApprovedByAdminMail;
use App\Notifications\User\AddMoney\RejectedByAdminMail;

class AddMoneyController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $page_title = __("All Logs");
        $transactions = Transaction::with(
            'user:id,email,username,full_mobile',
            'payment_gateway:id,name',
        )->where('type', 'add-money')->paginate(20);
        return view('admin.sections.add-money.index', compact(
            'page_title',
            'transactions'
        ));
    }


    /**
     * Pending Add Money Logs View.
     * @return view $pending-add-money-logs
     */
    public function pending()
    {
        $page_title = __("Pending Logs");
        $transactions = Transaction::with(
            'user:id,email,username,full_mobile',
            'payment_gateway:id,name',
        )->where('type', 'add-money')->where('status', 2)->paginate(20);
        return view('admin.sections.add-money.index', compact(
            'page_title',
            'transactions'
        ));
    }


    /**
     * Complete Add Money Logs View.
     * @return view $complete-add-money-logs
     */
    public function complete()
    {
        $page_title = __("Complete Logs");
        $transactions = Transaction::with(
            'user:id,username,email,username,full_mobile',
            'payment_gateway:id,name',
        )->where('type', 'add-money')->where('status', 1)->paginate(20);
        return view('admin.sections.add-money.index', compact(
            'page_title',
            'transactions'
        ));
    }

    /**
     * Canceled Add Money Logs View.
     * @return view $canceled-add-money-logs
     */
    public function canceled()
    {
        $page_title = __("Canceled Logs");
        $transactions = Transaction::with(
            'user:id,username,email,username,full_mobile',
            'payment_gateway:id,name',
        )->where('type', 'add-money')->where('status', 4)->paginate(20);
        return view('admin.sections.add-money.index', compact(
            'page_title',
            'transactions'
        ));
    }

    public function addMoneyDetails($id)
    {
        $data = Transaction::where('id', $id)->with(
            'user:id,firstname,lastname,email,username,full_mobile',
            'gateway_currency:id,name,alias,payment_gateway_id,currency_code,rate',
            'payment_gateway:type'
        )->where('type', 'add-money')->first();
        $page_title = __("Add money details for") . '  ' . $data->trx_id;
        return view('admin.sections.add-money.details', compact(
            'page_title',
            'data'
        ));
    }
    public function approved(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $data = Transaction::where('id', $request->id)->where('status', 2)->where('type', 'add-money')->first();
        try {
            //update wallet
            $userWallet = UserWallet::find($data->wallet_id);
            $userWallet->balance +=  $data->request_amount;
            $userWallet->save();
            //update transaction
            $data->status = 1;
            $data->available_balance =  $userWallet->balance;
            $data->save();
            $user = User::where('id', $data->user_id)->first();
            $user->notify(new ApprovedByAdminMail($user,$data));
            $notification_content = [
                'title'   => "Payment Approved",
                'message' => "Admin approved your payment",
                'time'    => Carbon::now()->diffForHumans(),
                'image'   => files_asset_path('profile-default'),
            ];
            UserNotification::create([
                'type'    => NotificationConst::RECHARGE_COMPLETE,
                'user_id' => $user->id,
                'message' => $notification_content,
            ]);
            // Push Notifications
            event(new UserNotificationEvent($notification_content,$user));
            send_push_notification(["user-".$user->id],[
                'title'     => $notification_content['title'],
                'body'      => $notification_content['message'],
                'icon'      => $notification_content['image'],
            ]);
            if ($data->invoice) {
                try{
                    SendSms($user,'RECHARGE_INVOICE','',['trx_id' => $data->trx_id,'request_amount' => $data->request_amount,'request_currency' => $data->request_currency, 'payable_amount'=> $data->total_payable, 'payable_currency'=>$data->payment_currency ]);
                }
                catch(Exception $e){}
            }
            return redirect()->back()->with(['success' => ['Add Money request approved successfully']]);
        } catch (Exception $e) {
            return back()->with(['error' => [$e->getMessage()]]);
        }
    }
    public function rejected(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
            'reject_reason' => 'required|string|max:200',
        ]);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $data = Transaction::where('id', $request->id)->where('status', 2)->where('type', 'add-money')->first();
        $data->status = 4;
        $data->reject_reason = $request->reject_reason;
        try {
            $data->save();
            $user = User::where('id', $data->user_id)->first();
            $user->notify(new RejectedByAdminMail($user,$data));
            $notification_content = [
                'title'   => "Payment Rejected",
                'message' => "Admin Rejected your payment",
                'time'    => Carbon::now()->diffForHumans(),
                'image'   => files_asset_path('profile-default'),
            ];
            UserNotification::create([
                'type'    => NotificationConst::RECHARGE_REJECT,
                'user_id' => $user->id,
                'message' => $notification_content,
            ]);
            //Push Notifications
            event(new UserNotificationEvent($notification_content,$user));
            send_push_notification(["user-".$user->id],[
                'title'     => $notification_content['title'],
                'body'      => $notification_content['message'],
                'icon'      => $notification_content['image'],
            ]);

            try {
                sendSms($user,'MONEY_RECHARGE_REJECT','',['amount'=> $data->request_amount,'request_currency'=>$data->request_currency,'gateway_name'=> $data->gateway_currency->gateway->name,'rejection_message'=>$request->reject_reason,'trx'=>$data->trx_id]);
            } catch (Exception $e) {}

            return redirect()->back()->with(['success' => ['Add Money request rejected successfully']]);
        } catch (Exception $e) {
            return back()->with(['error' => [$e->getMessage()]]);
        }
    }
    public function search(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'text'  => 'required|string',
        ]);
        if ($validator->fails()) {
            $error = ['error' => $validator->errors()];
            return Response::error($error, null, 400);
        }
        $validated = $validator->validate();

        $transactions = Transaction::with(
            'user:id,firstname,email,username,mobile',
            'payment_gateway:id,name',
        )->where('type', 'add-money')->where("trx_id", "like", "%" . $validated['text'] . "%")->latest()->paginate(20);
        return view('admin.components.data-table.add-money-transaction-log', compact(
            'transactions'
        ));
    }
}
